﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Collections.Concurrent;
using StockSharp.BusinessEntities;
using StockSharp.Algo.Storages;
using StockSharp.Algo.History;
using StockSharp.Algo.History.Russian.Rts;
using StockSharp.Messages;

namespace StorageTest {
   
    public class MySecStorage : ISecurityStorage {
        private ConcurrentDictionary<string, Security> storage = new ConcurrentDictionary<string, Security>();

        public IEnumerable<string> GetSecurityIds() {
            return storage.Keys;
        }

        public void Save(Security security) {
            storage.TryAdd(security.Id, security);
        }

        public object GetNativeId(Security security) {
            return security.Id;
        }

        public IEnumerable<Security> Lookup(Security criteria) {
            if (storage.ContainsKey(criteria.Id)) {
                return new Security[] { storage[criteria.Id] };
            } else {
                return Enumerable.Empty<Security>();
            }
        }

        public ConcurrentDictionary<string, Security> Dictionary { get { return storage; } }
    }

    class Program {
        StorageRegistry storageRegistry;
        MySecStorage secStorage = new MySecStorage();

        Program() {
            storageRegistry = new StorageRegistry();
            LocalMarketDataDrive d = (LocalMarketDataDrive)storageRegistry.DefaultDrive;
            d.Path = @"S:\Hydra2";
        }

        void loadRTS() {
            DateTime start = new DateTime(2015, 3, 17);
            RtsHistorySource src = new RtsHistorySource() { DumpFolder = @"S:\Forts", IsSystemOnly = true, LoadEveningSession = true };
            Parallel.For(0, 2, new ParallelOptions { MaxDegreeOfParallelism = 4 }, (i) => {
                DateTime date = start.AddDays(i);
                Console.WriteLine("Downloading {0}", date);
                var trades = src.LoadTrades(secStorage, date).Where(pair => pair.Key.Type == SecurityTypes.Future);
                lock (storageRegistry) {
                    Console.WriteLine("Storing {0}", date);
                    foreach (var pair in trades) {
                        var storage = storageRegistry.GetTradeStorage(pair.Key);
                        storage.AppendOnlyNew = true; // На всякий случай
                        storage.Save(pair.Value);
                    }
                }
            });
        }

        void checkLoaded() {
            foreach (var sec in secStorage.Dictionary
                    .OrderBy(pair => pair.Key)
                    .Select(pair => pair.Value)
                    .Where(sec => sec.Type == SecurityTypes.Future)) {
                Console.WriteLine("Processing {0}", sec);
                var tradeStorage = storageRegistry.GetTradeStorage(sec);
                foreach (var date in tradeStorage.Dates) {
                    Console.WriteLine("Loading {0}", date);
                    tradeStorage.Load(date).Last();
                }
            }
        }

        static void Main(string[] args) {
            Program p = new Program();
            p.loadRTS();
            p.checkLoaded();
        }
    }
}
