﻿using System.Windows;
using StockSharp.Quik;
using Ecng.Xaml;
using StockSharp.BusinessEntities;

namespace StockSharpWpf
{
	using System;

	using Ecng.Collections;
	using Ecng.Common;

	using StockSharp.Algo.Logging;

	/// <summary>
    /// Логика взаимодействия для MainWindow.xaml
    /// </summary>
	public partial class MainWindow : ILogReceiver
    {
        public MainWindow()
        {
            InitializeComponent();
        	InitializeLoger();
        }

		private void InitializeLoger()
		{
			try
			{
				_logManager.Listeners.Add(new FileLogListener("log.txt"));
				_logManager.Sources.Add(this);
				_logManager.Sources.Add(new UnhandledExceptionSource());
			}
			catch (Exception ex)
			{
				MessageBox.Show(ex.Message);
			}
		}


        //обьявляем переменную
        private QuikTrader _trader;

		private readonly LogManager _logManager = new LogManager();


        /// <summary>
        /// Подключение к квику
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void Connect_Click(object sender, RoutedEventArgs e)
        {
            //создаём квик трейдера , передавая ему сразу место расположения нашего терминала
            _trader= new QuikTrader(QuikTerminal.GetDefaultPath());

			_logManager.Sources.Add(_trader);

            //подписываемся на событие появление новых инструментов
            //оно сработает когда включится экспорт 
            _trader.NewSecurities += securities =>this.GuiAsync(()=>//GuiASync используется чтобы выводить графику из другого потока
                                                                    {
                                                                    	foreach (var security in securities)
                                                                    	{
                                                                    		this.AddInfoLog("Получен инструмент: {0}", security.Id);
                                                                    	}
                                                                        //заполняем коллекцию у нашего выпадающего списка (ComboBox)
                                                                        Securitites.ItemsSource = _trader.Securities;
                                                                    });
            //подписываемся на событие появления новых портфелей
            //сработает после запуска экспорта
            _trader.NewPortfolios += portfolios =>this.GuiAsync(()=>
                                                                    {
																		foreach (var p in portfolios)
																		{
																			this.AddInfoLog("Получен портфель: {0}", p.Name);
																		}
                                                                        //заполняем коллекцию у нашего выпадающего списка (ComboBox)
                                                                        Portfolios.ItemsSource = _trader.Portfolios;
                                                                    });

			//подписываемся на событие подключения, как только подлючимся, сразу запустим Экспорт
			//Connect- просто подключение к потоку
			//StartExport- получение он-лайн данных из квика Инструменты,Заявки , Портфели и так далее
			_trader.Connected += () =>
			{
				this.AddInfoLog("Соединение установлено, старт экспорта");

				try
				{
					_trader.StartExport();
				}
				catch (ArgumentException ex)
				{
					this.AddErrorLog(ex.Message);
				}
			};

			_trader.Disconnected += () => this.AddInfoLog("Соединение отключено");
			_trader.ProcessDataError += ex => this.AddWarningLog("Ошибка обработки: {0}".Put(ex.ToString()));
			_trader.ConnectionError += ex => this.AddWarningLog("Ошибка соединения: {0}".Put(ex.ToString()));
        	_trader.OrdersRegisterFailed += fail => fail.ForEach(f => this.AddInfoLog("Ошибка регистрации заявки: {0}", f.Error));
			_trader.OrdersCancelFailed += fail => fail.ForEach(f => this.AddInfoLog("Ошибка отмены заявки: {0}", f.Error));
        	_trader.NewOrders += orders => orders.ForEach(order => this.AddInfoLog("Новая заявка {0} ({1}) на {2} {3} с ценой {4} и объемом {5}/{6}", order.Id, order.TransactionId, order.Security.Code, order.Direction, order.Price, order.Volume, order.Balance));
        	_trader.OrdersChanged += orders => orders.ForEach(order => this.AddInfoLog("Заявка изменена {0} ({1}) на {2} {3} с ценой {4} и объемом {5}/{6}", order.Id, order.TransactionId, order.Security.Code, order.Direction, order.Price, order.Volume, order.Balance));
            
			// подключаем квик 
            _trader.Connect();

        }

        private void Buy_Click(object sender, RoutedEventArgs e)
        {
            if (_trader != null)
            {
            	try
            	{
					//создаём ордер
					//заполняем его нужными свойствами
					// портфель и инструмент вынимаем из выпдающих списков
					//цену для отправки заявки вынимаем из тесктового окна
					var order = new Order
					{
						Trader = _trader,
						Portfolio = (Portfolio)Portfolios.SelectedItem,
						Security = (Security)Securitites.SelectedItem,
						Volume = 1,
						Price = decimal.Parse(PRICE.Text),
						Direction = OrderDirections.Buy
					};

            		this.AddInfoLog("Регистрация заявки {0}, {1} {2} по {3} на {4}", order.Portfolio.Name, order.Direction, order.Security.Id, order.Price, order.Volume);

					//регистрируем орде
					_trader.RegisterOrder(order);
            	}
            	catch (Exception excp)
            	{
					this.AddErrorLog("Register order error: {0}", excp);
            	}
            }
        }

        private void CancelOrders_Click(object sender, RoutedEventArgs e)
        {
            //отменить все заявки
            if(_trader!=null)
            _trader.CancelOrders();
        }

        protected override void OnClosing(System.ComponentModel.CancelEventArgs e)
        {   //отключаемся от кивка на случай закрытия окна
            if(_trader!=null)
            {
             _trader.StopExport();
             _trader.Disconnect();
            }
            base.OnClosing(e);
        }

		#region Implementation of ILogSource

		private readonly Guid _id = Guid.NewGuid();


		/// <summary>
		/// Уникальный идентификатор источника.
		/// </summary>
		public Guid Id
		{
			get { return _id; }
		}

		/// <summary>
		/// Родительский источник логов.
		/// </summary>
		public ILogSource Parent
		{
			get { return null; }

		}

		/// <summary>
		/// Дочерние источники логов.
		/// </summary>
		public INotifyList<ILogSource> Childs
		{
			get { return null; }

		}

		/// <summary>
		/// Имя источника.
		/// </summary>
		string ILogSource.Name
		{
			get { return "MainWindow"; }
		}


		public event Action<LogMessage> Log;

		/// <summary>
		/// Записать сообщение в лог.
		/// </summary>
		/// <param name="message">Отладочное сообщение.</param>
		public void AddLog(LogMessage message)
		{
			Log.SafeInvoke(message);

		}

		#endregion
    }
}
