﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows;
using System.Windows.Media;

using Ecng.Xaml;
using Ecng.Collections;

using StockSharp.Algo.Candles;
using StockSharp.Algo.Equity;
using StockSharp.Algo.Logging;
using StockSharp.Algo.Reporting;
using StockSharp.Algo.Testing;
using StockSharp.BusinessEntities;

namespace Optimizator
{
	/// <summary>
	/// Interaction logic for RunStrategyWindow.xaml
	/// </summary>
	public partial class RunStrategyWindow : Window
	{
		private SmaStrategy.SmaStrategy _strategy;

		private ICollection<EquityData> _curveItems;
		private EmulationTrader _trader;

		private readonly LogManager _logManager = new LogManager();

		private DateTime _lastUpdateDate;

		public RunStrategyWindow(DateTime startTime, DateTime stopTime, StockSharp.Algo.Storages.TradingStorage storage, TimeSpan timeFrame, Optimization.Parameters Parametrs)
		{
			InitializeComponent();

			//_logManager.Listeners.Add(new FileLogListener("log.txt"));

			// создаем тестовый инструмент, на котором будет производится тестирование
			var security = new Security
			{
				Id = "RIU9@RTS",
				Code = "RIU9",
				Name = "RTS-9.09",
				MinStepSize = 5,
				MinStepPrice = 2,
				Decimals = 0,
				Exchange = Exchange.Test,
			};

			// тестовый портфель
			var portfolio = new Portfolio { Name = "test account" };

			// создаем шлюз для эмуляции
			// инициализируем настройки (инструмент в истории обновляется раз в секунду)
			_trader = new EmulationTrader(
				new[] { security },
				new[] { portfolio },
				storage)
			{
				MarketTimeChangedInterval = timeFrame,
				Storage = storage,
				WorkingTime = Exchange.Rts.WorkingTime,
				// параметр влияет на занимаемую память.
				// в случае достаточно количества памяти на компьютере рекомендуется его увеличить
				DaysInMemory = 6,// TODO Стоит вынести в настройки?
			};

			// * В стратегии стакан не используется? Сильно влияет на скорость?
			// * TODO Создание стакана вынести в виде флага на интерфейс
			_trader.DepthGenerators[security] = new TrendMarketDepthGenerator(security)
			{
				// стакан для инструмента в истории обновляется раз в секунду
				Interval = TimeSpan.FromSeconds(1),
			};

			// создаем торговую стратегию
			_strategy = new SmaStrategy.SmaStrategy();
			_strategy.Volume = 1;
			_strategy.Security = security;
			_strategy.Portfolio = portfolio;
			_strategy.Trader = _trader;

			// Устанавливаем параметры стратегии
			(_strategy as SmaStrategy.SmaStrategy).Parametrs.Set(Parametrs);

			// копируем параметры EquityManager на визуальную панель
			this.ParametersPanel.Parameters.Clear();
			this.ParametersPanel.Parameters.AddRange(_strategy.EquityManager.Parameters);

			_strategy.EquityManager.NewEquityData += data => this.GuiAsync(() => _curveItems.Add(data));

			_logManager.Sources.Add(_strategy);

			// и подписываемся на событие изменения времени, чтобы обновить ProgressBar
			_trader.MarketTimeChanged += () =>
			{
				// в целях оптимизации обновляем ProgressBar только при начале нового дня
				if (_trader.MarketTime.Date != _lastUpdateDate || _trader.MarketTime >= stopTime)
				{
					_lastUpdateDate = _trader.MarketTime.Date;
					this.GuiAsync(() => this.TestingProcess.Value = (_trader.MarketTime - startTime).Ticks);
				}
			};

			_trader.StateChanged += () =>
			{
				if (_trader.State == EmulationStates.Stopped)
				{
					this.GuiAsync(() =>
					{
						this.Report.IsEnabled = true;

						if (this.TestingProcess.Value == this.TestingProcess.Maximum)
						{
							var StrategyResults = new Optimization.StrategyResults(_strategy.GetType(), "dfd", startTime, stopTime);
							StrategyResults.Add(_strategy);
						}
						else
							MessageBox.Show("Отменено");
					});
				}
				else if (_trader.State == EmulationStates.Started)
				{
					// запускаем стратегию когда эмулятор запустился
					_strategy.Start();
				}
			};

			if (_curveItems == null)
				_curveItems = this.Curve.CreateCurve(_strategy.Name, Colors.DarkGreen);
			else
				_curveItems.Clear();

			// устанавливаем в визуальный элемент ProgressBar максимальное количество итераций)
			this.TestingProcess.Maximum = (stopTime - startTime).Ticks;
			this.TestingProcess.Value = 0;
			this.Report.IsEnabled = false;

			// соединяемся с трейдером и запускаем экспорт,
			// чтобы инициализировать переданными инструментами и портфелями необходимые свойства EmulationTrader
			_trader.Connect();
			_trader.StartExport();

			// запускаем эмуляцию
			_trader.Start(startTime, stopTime);
		}

		private void Report_Click(object sender, RoutedEventArgs e)
		{
			// сгерерировать отчет по прошедшему тестированию
			// Внимание! сделок и заявок может быть большое количество,
			// поэтому Excel отчет может тормозить
			new ExcelStrategyReport(_strategy, "sma.xlsx").Generate();



			// открыть отчет
			Process.Start("sma.xlsx");
		}
	}
}
