﻿namespace StockSharp.AlfaDirect
{
	using System;
	using System.Globalization;

	using ADLite;

	using Ecng.Common;
	using Ecng.ComponentModel;

	using StockSharp.BusinessEntities;

	class AlfaWrapper
	{
		const string _marketDepthTable = "queue"; // Стаканы
		const string _marketDepthFields = "sell_qty, price, buy_qty";

		const string _securityTable = "fin_info"; // Инструменты
        const string _securityFieldsCreate = "paper_no, ANSI_name, mat_date, status, p_code, place_code, curr_code, go_buy, go_sell, board_code, open_pos_qty";
        const string _securityFieldsUpdate = "paper_no, sell, sell_qty, buy, buy_qty, min_deal, max_deal, open_pos_qty";

		//private const string SecurityFields = "paper_no, ts_p_code, board_code, sell, sell_qty, buy, buy_qty, min_deal, max_deal, lot_size, decimals, status, price_step, p_code, volatility, theor_price, mat_date";

		private const string _portfolioTable = "acc_bal"; // БАЛАНС ПОЗИЦИЙ ПО СЧЕТАМ
		private const string _portfolioFields = "treaty, forward_bal, money";

		private const string _tradeTable = "all_trades"; // Текущие сделки на рынке
		private const string _tradeFields = "trd_no, paper_no, qty, price, ts_time, b_s";

		private const string _orderTable = "orders";
		private const string _orderFields = "ord_no, acc_code, paper_no, status, b_s, price, qty, rest, ts_time, comments";

		private const string _positionsTable = "sum_balance";
		private const string _positionsFields = "treaty, paper_no, netto_rest, p_code";

		private const string _myTradesTable = "trades"; // сделки
		private const string _myTradesFields = "trd_no, ord_no, treaty, paper_no, price, qty, b_s, ts_time";

		//private const string _logTag = "AlfaWrapper";

		private readonly AlfaDirectClass _ad;

		public AlfaWrapper()
		{
			_ad = new AlfaDirectClass();
			_ad.OnConnectionChanged += ConnectionChanged;
			_ad.OnTableChanged += TableChanged;
		}

		public string Version
		{
			get { return (string)_ad.Version; }
		}

		public bool IsConnected
		{
			get { return _ad.Connected; }
		}

		public DateTime MarketTime
		{
			get { return _ad.SessionTime; }
		}

		public event Action<string, bool> ProcessSecurities;
		public event Action<string, string> ProcessTrades;
		public event Action<string, string> ProcessQuotes;
		public event Action<string> ProcessPortfolios;
		public event Action<string> ProcessOrders;
		public event Action<string> ProcessPositions;
		public event Action<string> ProcessMyTrades;

		public event Action Connected;
		public event Action Disconnected;
		public event Action<Exception> ConnectionError;

		private void TableChanged(string tableName, string tableParams, object data, object fieldtypes)
		{
			//Logger.Debug("TableChanged: " + tableName + "; params: " + tableParams, _logTag);

			switch (tableName)
			{
				case _marketDepthTable:
					ProcessQuotes.SafeInvoke(tableParams, (string)data);
					break;
				case _tradeTable:
					ProcessTrades.SafeInvoke(tableParams, (string)data);
					break;
				case _securityTable:
					ProcessSecurities.SafeInvoke((string)data, false);
					break;
				case _portfolioTable:
					ProcessPortfolios.SafeInvoke((string)data);
					break;
				case _orderTable:
					ProcessOrders.SafeInvoke((string)data);
					break;
				case _positionsTable:
					ProcessPositions.SafeInvoke((string)data);
					break;
			}
		}

		private void ConnectionChanged(eConnectionState state)
		{
			switch (state)
			{
				case eConnectionState.Connected:
					Connected.SafeInvoke();
					break;
				case eConnectionState.Disconnected:
					Disconnected.SafeInvoke();
					break;
				default:
					ConnectionError.SafeInvoke(new Exception("Error eConnectionState: " + state));
					break;
			}
		}

		public void StartExportOrders()
		{
			RegisterTable(_orderTable, _orderFields, string.Empty);
		}

		public void StopExportOrders()
		{
			UnRegisterTable(_orderTable, string.Empty);
		}

		public void StartExportPositions()
		{
			var data = GetLocalDbData(_positionsTable, _positionsFields);
			ProcessPositions.SafeInvoke(data);

			RegisterTable(_positionsTable, _positionsFields, string.Empty);
		}

		public void StopExportPositions()
		{
			UnRegisterTable(_positionsTable, string.Empty);
		}

		public void RegisterPortfolio(Portfolio portfolio)
		{
			RegisterTable(_portfolioTable, _portfolioFields, GetFilter(portfolio));
		}

		public void RegisterSecurity(Security security)
		{
			RegisterTable(_securityTable, _securityFieldsUpdate, GetFilter(security));
		}

		public void RegisterQuotes(Security security)
		{
			RegisterTable(_marketDepthTable, _marketDepthFields, GetFilter(security));
		}

		public void RegisterTrades(Security security)
		{
			RegisterTable(_tradeTable, _tradeFields, GetFilter(security));
		}

		public void UnRegisterSecurity(Security security)
		{
			UnRegisterTable(_securityTable, GetFilter(security));
		}

		public void UnRegisterTrades(Security security)
		{
			UnRegisterTable(_tradeTable, GetFilter(security));
		}

		public void UnRegisterPortfolio(Portfolio portfolio)
		{
			if (portfolio == null)
				throw new ArgumentNullException("portfolio");

			UnRegisterTable(_portfolioTable, GetFilter(portfolio));
		}

		public void UnRegisterQuotes(Security security)
		{
			UnRegisterTable(_marketDepthTable, GetFilter(security));
		}

		private static string GetFilter(Portfolio portfolio)
		{
			if (portfolio == null)
				throw new ArgumentNullException("portfolio");

			return "treaty = " + portfolio.Name;
		}

		private static string GetFilter(Security security)
		{
			if (security == null)
				throw new ArgumentNullException("security");

			return "paper_no = " + security.GetPaperNo();
		}

		public void RegisterTable(string tableName, string fields, string where)
		{
			if (tableName.IsEmpty())
				throw new ArgumentNullException("tableName");

			string message;
			var code = _ad.SubscribeTable(tableName, fields, where, eSubsctibeOptions.UpdatesOnly, out message);

			if (code != tagStateCodes.stcSuccess)
				throw new AlfaException(code, message);
		}

		public void UnRegisterTable(string tableName, string where)
		{
			if (tableName.IsEmpty())
				throw new ArgumentNullException("tableName");

			string message;
			var code = _ad.UnSubscribeTable(tableName, where, out message);

			if (code != tagStateCodes.stcSuccess)
				throw new AlfaException(code, message);
		}

		public void RegisterOrder(Order order)
		{
			//Logger.Debug("RegisterOrder", _logTag);

			if (order == null)
				throw new ArgumentNullException("order");

			var account = order.Portfolio.Name + "-000"; // Портфель. // TODO: use AD portfolio instead of the AD account, "-000" is hardcoded
			var placeCode = order.Security.GetPlaceCode(); // Код торговой площадки (рынка).
			var pCode = order.Security.Code; // Код финансового инструмента.
			var endDate = AlfaUtils.CancelTimeToAlfa(order.CancelTime); // Срок действия поручения.
			var comments = order.Comment; // Комментарий.
			var currency = order.Security.GetCurrency(); // Валюта цены.
			var buySell = AlfaUtils.OrderDirectionsToAlfa(order.Direction); // Покупка/Продажа
			var quantity = (int)order.Volume; // Количество.
			var price = order.Price; // Цена.

			//Logger.Debug(account + ", " + placeCode + ", " + pCode + ", " + endDate + ", " + comments + ", " + currency + ", " +
			//				buySell + ", " + quantity + ", " + price);

			if (order.Type == OrderTypes.Conditional)
			{
				var condition = (AlfaStopCondition)order.StopCondition;
				var slippage = condition.Slippage; // Проскальзывание
				var takeProfit = condition.TakeProfit; // Цена фиксирования прибыли

				//Logger.Debug(slippage + ", " + takeProfit, _logTag);

				_ad.CreateStopOrder(account, placeCode, pCode, endDate, comments,
									currency, buySell, quantity, (double)price, slippage, takeProfit, -1);
			}
			else
			{
				if (order.Type == OrderTypes.Market)
				{
					var details = GetLocalDbData(_securityTable, _securityFieldsUpdate, GetFilter(order.Security));

					if (details.IsEmpty())
					{
						//Logger.Error("Ошибка расчета цены для маркет-заявки", _logTag);
						order.State = OrderStates.Failed;
						return;
					}

					var data = details.Split('|');

					if (order.Direction == OrderDirections.Buy)
					{
						// best ask + 100 min price steps
						price = data[1].To<decimal>() + 100 * order.Security.MinStepSize;
					}
					else
					{
						// best bid - 100 min price steps
						price = data[3].To<decimal>() - 100 * order.Security.MinStepSize;
					}
				}

				_ad.CreateLimitOrder(account, placeCode, pCode, endDate, comments, currency, buySell, quantity,
									 (double)price, null, null, null, null, null, "Y", null, null, null, null, null, null, null, null, null, null, -1);
			}
		}

		public void ReRegisterOrder(Order oldOrder, Order newOrder)
		{
			//Logger.Debug("ReRegisterOrder(): id: " + oldOrder.Id + ", price: " + 
			//    newOrder.Price + ", vol.: " + newOrder.Volume, _logTag);

			//if (oldOrder.Id == 0)
			//{
			//    Logger.Error("Failed to re-register order, order id is 0", _logTag);
			//    return;
			//}

			var endDate = AlfaUtils.CancelTimeToAlfa(oldOrder.CancelTime);
			_ad.UpdateLimitOrder((int)oldOrder.Id, newOrder.Price, newOrder.Volume, endDate, -1);
		}

		public void CancelOrder(Order order)
		{
			if (order == null)
				throw new ArgumentNullException("order");

			_ad.DropOrder(order.Id, null, null, null, null, null, -1);
		}

		public void CancelOrders(bool? isStopOrder, Portfolio portfolio, OrderDirections? direction, string classCode, Security security)
		{
			var isBuySell = direction == null ? null : (direction == OrderDirections.Buy ? "B" : "S");
			_ad.DropOrder(null, isBuySell, portfolio == null ? null : portfolio.Name, null, classCode, security == null ? null : security.GetPaperNo(), -1);
		}

		/// <summary>
		/// Загрузить все доступные инструменты.
		/// </summary>
		public void ReadSecurities()
		{
			//Logger.Debug("ReadSecurities()", _logTag);

			var data = GetLocalDbData(_securityTable, _securityFieldsCreate);
			ProcessSecurities.SafeInvoke(data, true);
		}

		/// <summary>
		/// Загрузить все доступные портфели.
		/// </summary>
		public void ReadPortfolios()
		{
			//Logger.Debug("ReadPortfolios()", _logTag);

			var data = GetLocalDbData(_portfolioTable, _portfolioFields);
			ProcessPortfolios.SafeInvoke(data);
		}

		/// <summary>
		/// Загрузить список всех заявок.
		/// </summary>
		public void ReadOrders()
		{
			//Logger.Debug("ReadOrders()", _logTag);

			var data = GetLocalDbData(_orderTable, _orderFields);

			if (data == null) return;

			ProcessOrders.SafeInvoke(data);
		}

		/// <summary>
		/// Загрузить список всех сделок.
		/// </summary>
		public void ReadMyTrades()
		{
			//Logger.Debug("ReadMyTrades()", _logTag);

			var data = GetLocalDbData(_myTradesTable, _myTradesFields);

			if (data == null) return;

			ProcessMyTrades.SafeInvoke(data);
		}

		string GetLocalDbData(string table, string fields, string where = "")
		{
			return _ad.GetLocalDBData(table, fields, where);
		}

		/// <summary>
		/// Получить информацию о минимальном шаге и его стоимости.
		/// </summary>
		/// <param name="security">Инструмент.</param>
		/// <param name="step">Минимальный шаг цены.</param>
		/// <param name="cost">Стоимость шага цены.</param>
		public void GetPriceStepInfo(Security security, out decimal step, out decimal cost)
		{
			step = 0;
			cost = 0;

			const string table = "papers";
			const string fields = "price_step, price_step_cost";
			var where = GetFilter(security);

			var data = GetLocalDbData(table, fields, where);

			if (data == null)
			{
				//Logger.Error("Failed to get price step info for the " + security.Code + ", data is null");
				return;
			}

			var details = data.Split('|');

			//try
			//{
			step = Decimal.Parse(details[0], NumberStyles.AllowExponent | NumberStyles.AllowDecimalPoint);
			cost = Decimal.Parse(details[1], NumberStyles.AllowExponent | NumberStyles.AllowDecimalPoint);
			//}
			//catch (Exception ex)
			//{
			//	Logger.Error("Failed to read price step info for the " + security.Code + ": " + ex.Message, _logTag);
			//}
		}

		public string GetExchangeCode(string placeCode)
		{
			const string table = "trade_places";
			const string fields = "place_code, place_name, ex_code";
			var where = "place_code = " + placeCode;

			var data = GetLocalDbData(table, fields, where);

			return data.IsNull() ? String.Empty : data.Split('|')[2];
		}

		public string GetHistoryData(Security security, AlfaTimeFrames timeFrame, Range<DateTime> range)
		{
			//Logger.Debug("GetHistoryData: security = " + security.Id, _logTag);

			const int modeStoreLocal = 2;
			const int maxWaitTimeSec = 10 * 60;

			var data = (string)_ad.GetArchiveFinInfo(security.GetPlaceCode(), security.Code,
				timeFrame.Interval, range.Min, range.Max, modeStoreLocal, maxWaitTimeSec);

			return data;
		}
	}
}