﻿using System;
using System.IO;
using System.Threading;
using System.Windows;
using System.Collections.ObjectModel;

using Ecng.Trading.Quik;
using Ecng.Common;
using Ecng.Trading.BusinessEntities;
using Ecng.Xaml;
using Ecng.Trading.Algo;

namespace GUI
{
	public partial class MainWindow : Window
	{
		// подключение в Quik
		private QuikTrader trader;

		// лог файл
		private StreamWriter writer;

		// инструмент, по которому будут выставляться заявки
		private Security lkoh;

		// порфтель, по которому будут выставляться заявки
		private Portfolio micexPf;

		// список заявок, в который будут добавляться новые заявки
		private ObservableCollection<Order> orders = new ObservableCollection<Order>();

		public MainWindow()
		{
			InitializeComponent();

			// открываем файл на добавление в конец (если файл не создан, то он создается)
			writer = File.AppendText("robot.txt");

			// отключаем буферизацию (все записи будут автоматически попадать на диск)
			writer.AutoFlush = true;

			// выводим сообщение о времени запуска
			writer.WriteLine("Стартовал в " + DateTime.Now);

			// привязываем список заявок к таблице
			// теперь при добавлении в список новой заявки, она автоматически попадет в таблицу
			OrdersDetails.ItemsSource = orders;
		}

		private void Connect_Click(object sender, RoutedEventArgs e)
		{
			// если подключение уже было ранее создано
			if (trader != null)
			{
				// вывести об этом предупреждение
				MessageBox.Show("Повторное подключение.");

				// и выйти из метода
				return;
			}

			// создаем новое подключение к Quik
			// путь к терминалу берется из формы ввода
			trader = new QuikTrader(QuikPath.Text);

			// подписываемся на событие успешного подключения
			trader.Connected += () =>
			{
				// так как наше событие было вызвано в параллельном потоке,
				// и нам требуется обратиться к визуальному элементу,
				// то необходимо с синхронизоваться с GUI через метод GuiSync
				this.GuiSync(() =>
				{
					// выводим сообщение на экран
					Notification.Content = "Подключилось!";
				});

				// запускаем экспорт
				trader.StartExport();
			};

			// подписываемся на событие появления новых инструментов
			// где ищем появления инструмента Лукойл
			trader.NewSecurities += securities =>
			{
				foreach (var security in securities)
				{
					AddLog(security.Code);

					if (security.Code == "LKOH")
					{
						lkoh = security;
						trader.RegisterQuotes(lkoh);
					}
				}
			};

			// подписываемся на событие появления новых портфелей
			// где ищем появления портфеля ММВБ
			trader.NewPortfolios += portfolios =>
			{
				foreach (var portfolio in portfolios)
				{
					AddLog(portfolio.Name);

					if (portfolio.Name == "NL0011100043")
						micexPf = portfolio;
				}
			};

			// подписываемся на событие появления ошибки
			trader.ProcessDataError += error =>
			{
				// так как наше событие было вызвано в параллельном потоке,
				// и нам требуется создать визуальный элемент (окно),
				// то необходимо с синхронизоваться с GUI через метод GuiSync
				this.GuiSync(() =>
				{
					// выводим текст ошибки на экран
					MessageBox.Show(error.ToString());
				});
			};

			trader.NewOrders += newOrders =>
			{
				// так как наше событие было вызвано в параллельном потоке,
				// и при добавлении в коллекцию orders добавляется строчка в визуальную таблицу заявок,
				// то необходимо с синхронизоваться с GUI через метод GuiSync
				this.GuiSync(() =>
				{
					foreach (var newOrder in newOrders)
					{
						orders.Add(newOrder);
					}
				});
			};

			// подключаемся
			trader.Connect();
		}

		// метод для вывода в лог
		private void AddLog(string text)
		{
			// в начале выводим в файл
			writer.WriteLine(text);

			// затем в текстовое поле на окне (так же нужна синхронизация)
			this.GuiSync(() =>
			{
				Log.Text = text + Environment.NewLine + Log.Text;
			});
		}

		// обработчик кнопки, по которой открывается окно для ввода цены и объема заявки
		private void NewOrder_Click(object sender, RoutedEventArgs e)
		{
			// создаем переменную окна и присваеваем ей значение типа NewOrderWindow
			NewOrderWindow wnd = new NewOrderWindow();

			// показываем окно пользователю
			wnd.ShowModal();

			// создаем заявку и инициализируем ее необходимыми параметрами
			Order order = new Order();
			order.Price = wnd.Price.Text.To<double>();
			order.Volume = wnd.Volume.Text.To<int>();
			order.Direction = OrderDirections.Buy;
			order.Security = lkoh;
			order.Portfolio = micexPf;

			// регистрируем заявку
			trader.RegisterOrder(order);
		}

		// обработчик кнопки Старт по которой запускается отдельный поток
		// 
		private void Start_Click(object sender, RoutedEventArgs e)
		{
			// создаем отдельный поток
			new Thread(() =>
			{
				// выставляется заявка на покупку на край спреда +0.01 копейка и переставляется до тех пор, пока не исполниться

				Order order = null;

				// пока наша заявка не зарегистрирована или не исполнилась
				while (order == null || order.State != OrderStates.Done)
				{
					// если наша заявка не зарегистрирована
					if (order == null)
					{
						// создаем заявку и инициализируем ее необходимыми параметрами
						order = new Order();
						order.Price = lkoh.BestBid.Price + 0.01;
						order.Security = lkoh;
						order.Volume = 1;
						order.Portfolio = micexPf;
						order.Direction = OrderDirections.Buy;

						// округляем цену, так как double значение может содержать неточное число
						order.Price = lkoh.ShrinkPrice(order.Price);

						// регистрируем заявку
						trader.RegisterOrder(order);
					}
					else // если уже была ранее зарегистрирована
					{
						// если цена заявки меньше или равна цене текущего бида
						if (order.Price <= lkoh.BestBid.Price)
						{
							// отменяем заявку
							trader.CancelOrder(order);

							// обнуляем переменную (чтобы цикл продолжился выполняться)
							order = null;
						}
					}

					// засыпаем на 1 секунду
					Thread.Sleep(1000);
				}

			}).Start();
		}
	}
}
